;******************************************************************************
;*                                                                            *
;*  Project :     	ingenia BootLoader                                        *
;*  Module :      	iBL.s                                                     *
;*  Description : 	dsPic bootloader with autobaud detection                  *
;*               	Read/Write through UART : PGM, EEPROM & Config registers  *
;*  Author :      	Roger Juanpere                                            *
;*                                                                            *
;*  Revision :	1.0 (17-08-05): Initial version                               *
;*            	1.1 (01-02-06): Added support for >32K PGM devices            *
;*			  	1.2 (05-02-07): Dennis modification for dsPic33FJ128GP306	  *
;*				1.3 (25-04-07): bra StartFrame instead of bra SendNACK when   *
;*								command code is unknown						  *
;*                                                                            *
;******************************************************************************
;*  ingenia-cat S.L. (c)   -   www.ingenia-cat.com                            *
;******************************************************************************
;*	Dennis has modified the project for :									  *
;*	1)  dsPIC30FJ128GP306 (include the p33FJ128GP306.gld)					  *
;*	2)  Using the main UART (not alternate UART) #0x8420 -> #0x8020			  *
;*	3)  Change to U2ART, all U1XXX => U2XXX									  *
;*	4)  Change to IC2, all IC1XXX => IC2XXX									  *
;*	2)  configuration bits + startup code for 40MHz							  *
;*	3)  set USER_ADDRESS to 0x0200											  *
;*	4)  set START_ADDRESS to 0x015200										  *
;*	5)  set MAX_WORD_ROW to 128												  *
;*	6)  remove EEPROM related codes											  *
;*	7)  remove NVMADRU														  *
;*  8)  use W5 instead of NVMADU for effective address						  *
;*  9)  modify NVMCON codes (stored in W0 or W8)							  *
;*  10) each row has 128 words for flash memory (stored in W3)				  *
;*  11) check if page erase is necessary									  *
;******************************************************************************

        .equ __33FJ128GP306, 1
        .include "/usr/pic30-elf/support/inc/p33FJ128GP306.inc"

;******************************************************************************
; Configuration bits:
;******************************************************************************

        config __FOSCSEL, FNOSC_FRCPLL 		;FRC Oscillator with PLL

        config __FWDT, FWDTEN_OFF           ;Watchdog Timer Enabled/disabled by user software
        
        config __FOSC, FCKSM_CSDCMD & OSCIOFNC_ON & POSCMD_NONE
        									;Clock Switching and Fail Safe Clock Monitor is disabled
											;OSC2 Pin Function: OSC2 is Digital I/O
											;Primary Oscillator Mode: Disabled

;******************************************************************************
; Program Specific Constants (literals used in code)
;******************************************************************************
          .equ CRC, W4
          .equ ACK, 0x55
          .equ NACK, 0xFF
          .equ USER_ADDRESS, 0x0200		; User's code start address
          .equ START_ADDRESS, 0x015200  ; Bootloader's start address located at
          								; last page of memory (0x015400 - 0x157FF, i.e. 1Kwords) 
          								; Relative to USER_ADDRESS
          .equ CFG_M, 0xF8				; Configuration memory TBLPAGE				

          .equ C_READ, 0x01
          .equ C_WRITE, 0x02
          .equ C_VERSION, 0x03
          .equ C_USER, 0x0F
          .equ MAX_WORD_ROW, 128		; write row operation, 64 instructions (256 bytes)

          .equ MAJOR_VERSION, 0x01
          .equ MINOR_VERSION, 0x03

;******************************************************************************
; Global Declarations:
;******************************************************************************
          .global __reset          ;The label for the first line of code.
          .global recBuf

;******************************************************************************
;Uninitialized variables in X-space in data memory
;******************************************************************************
          .section bss, xmemory
recBuf:   .space 2 * MAX_WORD_ROW  ; Store page data

;******************************************************************************
;Code Section in Program Memory
;******************************************************************************
        .text                     ; Start of Code section
        .org #START_ADDRESS
__reset:
        ; Config Target Speed to 40MHz
        mov #38, W0			      ; M=40: PLL Feedback Divisor bits
        mov W0, PLLFBD
        mov #0x0000, W0			  ; N1=2: PLL VCO Output Divider Select bits
        mov W0, CLKDIV			  ; N2=2: PLL Phase Detector Input Divider bits
        mov #0x0000, W0			  ; 
        mov W0, OSCTUN			  ; OSCTUN=0: Use centre frequency 7.37MHz
        mov #0x0021, W0			  ; 
        mov W0, RCON			  ; Disable Watchdog timer

        MOV #__SP_init, W15       ; Initialize the Stack Pointer
        MOV #__SPLIM_init, W0     ; Initialize the Stack Pointer Limit Register
        MOV W0, SPLIM
        NOP                       ; Add NOP to follow SPLIM initialization

WaitPLLLock:
        btss OSCCON, #LOCK        ; Wait until PLL has locked
        bra WaitPLLLock        

        ; Uart init
        mov #0x8020, W0           ; W0 = 0x8020 -> 1000 0000 0010 0000b
        mov W0, U2MODE            ; Enable UART, AutoBaud and 8N1
        clr U2STA

        ; Timer 3 init
        clr T3CON                 ; Stops any 16-bit Timer3 operation
        bclr IEC0, #T3IE          ; Disable Timer 3 interrupt
        setm PR3                  ; Set Timer 3 period to maximum value 0xFFFF
        mov #0x8000, W0           ; Start Timer 3 with 1:1 prescaler and clock source set to internal cycle
        mov W0, T3CON

        ; Input Capture init
        clr IC2CON                ; Turn off Input Capture 1 module
        bset IC2CON, #1           ; Input Capture Mode every risind edge
        bclr IFS0, #IC2IF         ; Clear Input Capture flag
        bclr IEC0, #IC2IE         ; Disable Input Capture interrupts

        ; Start Autobaud detection
        mov #0x0004, W0           ; W0 = 0x0004
        rcall WaitRising          ; Wait until the first Rising edge is detected
        clr TMR3                  ; Clear content of the Timer 3 timer register
ByteLoop:
        rcall WaitRising
        dec W0, W0                ; W0--
        bra NZ, ByteLoop          ; if W0 != 0 jump to ByteLoop
        bclr T3CON, #TON          ; Last Rising edge detected so Stop Timer 3
        mov TMR3, W0              ; W0 = TMR3
        add #0x40, W0             ; For rounding: +64 >> 7 is equal to +0.5
        asr W0, #7, W0            ; W0 = ((Tend - Tini + 64) / 128)
        dec W0, W0                ; W0--

        ; Uart re-init
        mov W0, U2BRG             ; U2BRG = W0 -> Configs UART with the detected baudrate
        bclr U2MODE, #ABAUD       ; Disable AutoBaud
        bset U2STA, #UTXEN        ; Enable transmition
        bra SendAck

StartFrame:
        btss U2STA, #URXDA        ; Wait until a character is received
        bra StartFrame
        mov U2RXREG, W0
        cp.b W0, #C_USER          ; Compare received Character with USER character
        btsc SR, #Z
        goto USER_ADDRESS
        cp.b W0, #C_READ          ; Compare received Character with READ character
        bra Z, ReadMemCmd
        cp.b W0, #C_WRITE         ; Compare received Character with WRITE character
        bra Z, WriteMemCmd
        cp.b W0, #C_VERSION       ; Compare received Character with VERSION character
        bra Z, VersionCmd
        bra StartFrame             

VersionCmd:
        mov #MAJOR_VERSION, W0    ; Send Major Version
        mov W0, U2TXREG
        mov #MINOR_VERSION, W0    ; Send Minor Version
        mov W0, U2TXREG
        bra SendAck

ReadMemCmd:
        rcall ReceiveChar         ; Receive high byte of the address
        mov W0, TBLPAG            ; High address byte
        rcall ReceiveChar         ; Receive medium byte of the address
        swap W0
        rcall ReceiveChar         ; Receive low byte of the address

        tblrdh [W0], W1           ; Read high word to W1
        mov W1, U2TXREG           ; Send W1 low byte

        tblrdl [W0], W1           ; Read low word to W1
        swap W1
        mov W1, U2TXREG           ; Send W1 high byte
        swap W1
        mov W1, U2TXREG           ; Send W1 low byte
SendAck:
        mov #ACK, W0              ; Send an ACK character
        bra Send
SendNack:
        mov #NACK, W0             ; Send a KO character
Send:
        mov W0, U2TXREG
        bra StartFrame

WriteMemCmd:
        clr W4                    ; Reset W4 = Checkbyte
        rcall ReceiveChar         ; Receive high byte of the initial address
        mov W0, TBLPAG            ; For latch loading and programming
        rcall ReceiveChar         ; Receive medium byte of the initial address
        swap W0			   
        rcall ReceiveChar         ; Receive low byte of the initial address
        mov W0, W5			      ; W5 = effective address
		tblwtl W0, [W0]           ; set address for erase cycle (if needed)

        rcall ReceiveChar         ; Receive the number of bytes to be received (high byte)
        swap W0
        rcall ReceiveChar		  ; Receive the number of bytes to be received (low byte)
        mov W0, W3
        mov #recBuf, W2           ; W2 = recBuf
FillBufLoop:
        rcall ReceiveChar
        mov.b W0, [W2++]          ; Move received byte to recBuf
        dec W3, W3
        bra nz, FillBufLoop       ; Fill reception buffer

        cp0.b W4                  ; Check (INTEL HEX8 Checksum - Sum modulo 256)
        bra nz, SendNack          ; if Checkbyte != 0 jump to SendNack
        mov #recBuf, W2           ; W2 = recBuf

        mov #CFG_M, W0            ; Check if destination is Config Memory
        cp.b TBLPAG
        bra nz, noCFM

CFM:
        mov #0x4000, W8           ; Assigns Write Config Row Code - Program a single Configuration register byte
        mov #1, W3                ; Assigns Number of 16bits words per Row
        bra LoadLatch

noCFM:
		mov W5, W0                ; Check if it is start of page
		mov #0x03FF, W8           ; page starts 0x0000, 0x00400, 0x00800, 0x00C00 etc
		and W8, W0, W0           
		cp0 W0
		bra nz, WriteRow          ; Write row

        mov #0x4042, W0           ; Assigns Erase PGM Page Code - Memory page erase operation (ERASE = 1)
        rcall WriteKey            ; Erase selected Page

WriteRow:
        mov #0x4001, W8           ; Assigns Write PGM Row Code - Memory row program operation (ERASE = 0)
        mov #128, W3              ; Assigns Number of 16bits word per Row (64instr - 128word16)

LoadLatch:
        tblwtl [W2++], [W5]       ; Load low word to latch
        dec W3, W3
        bra Z, EndLatch
        tblwth [W2++], [W5++]     ; Load high word to latch
        dec W3, W3                ; Repeat until whole row is loaded
        bra NZ, LoadLatch
EndLatch:
        mov W8, W0                ; Write selected Row
        rcall WriteKey
        bra SendAck               ; Send an ACK character

;******************************************************************************
;Procedures
;******************************************************************************
WaitRising:
        mov #0x5A, W2             ; W2 = 0x5A
MajorLRise:
        setm W1                   ; W1 = 0xFFFF
MinorLRise:
        btsc IFS0, #IC2IF         ; Rising edge detected?
        bra EndRising             ; Yes -> Jump to finish detection
        dec W1, W1                ; W1--
        bra NZ, MinorLRise        ; if W1 != 0 jump MinorLRise
        dec W2, W2                ; W2--
        bra NZ, MajorLRise        ; if W2 != 0 jump MajorLRise
        goto USER_ADDRESS         ; Timeout aprox. = 0x5A * 0xFFFF * 5 clocks -> Jump to user soft

EndRising:
        bclr IFS0, #IC2IF         ; Clear Interrupt Flag
        return

;******************************************************************************
ReceiveChar:
        mov #0xFFFF, W10          ; W10 = 0xFFFF
MajorLChar:
        setm W11                  ; W11 = 0xFFFF
MinorLChar:
        btsc U2STA, #URXDA        ; Character received ?
        bra EndReceiveChar        ; Yes -> Jump to Finish reception
        dec W11, W11              ; W1--
        bra NZ, MinorLChar        ; if W1 != 0 jump MinorLChar
        dec W10, W10              ; W2--
        bra NZ, MajorLChar        ; if W2 != 0 jump MajorLChar
        MOV #__SP_init, W15       ; Initialize Stack Pointer
        bra SendNack              ; Timeout aprox. = 0xFFFF * 0xFFFF * 5 clocks -> Jump to Send Nack
EndReceiveChar:
        mov.b U2RXREG, WREG       ; W0 = U2RXREG
        add.b W4, W0, W4          ; Checkbyte += W0 -> Performs a Sum modulo 256 checksum (INTEL HEX8)
        return

;******************************************************************************
WriteKey:
        mov W0, NVMCON
        mov #0x55, W0
        mov W0, NVMKEY
        mov #0xAA, W0
        mov W0, NVMKEY
        bset NVMCON, #WR          ; Start Writing
        nop
        nop
WaitWriting:
        btsc NVMCON, #WR          ; WR or WREN - Wait until operation is finished
        bra WaitWriting
        return

;--------End of All Code Sections ---------------------------------------------

.end                              ; End of program code in this file
