/**
 * @project		dsPicProgrammer Program
 * @description	dsPicProgrammer for Microchip dsPic with dsPicBootloader
 * @author			Dennis (dennis@amonics.com)
 ******************************************************************************   
 * Licensing:	GNU GENERAL PUBLIC LICENSE
 *				Version 2, June 1991
 *				Please refer to GNU license for details (license.txt)
 ******************************************************************************/

package memory;

/*****************************************************************************
 * Basic class for row memory
 *****************************************************************************/
public abstract class Memory {
	
	protected int start_addr = 0;		//Start address for flash row (word)
	public byte[] buffer = null;		//Buffer space for row
	private int counter = 0;			//byte pointer for buffer

	
	/***************************************************************************
	 * Add data to buffer 
	 * @param data add byte to buffer
	 * @return success/failure(buffer is full)
	 ***************************************************************************/	
	public boolean addData(byte data){
		/*
		 * Buffer is not full 
		 */
		if(counter < buffer.length){
			buffer[counter++] = data;
			return true;
		}
		/*
		 * Buffer is already full
		 */
		else{
			return false;
		}
	}

	/***************************************************************************
	 * Get the next pic address position to write
	 * @return next position (address) for writing
	 ***************************************************************************/
	public int getStartAddr(){
		return start_addr;
	}

	
	/***************************************************************************
	 * Get the next pic address position to write
	 * @return next position (address) for writing
	 ***************************************************************************/
	public int getNextPosition(){
		return (start_addr + counter/2);
	}
	
	/***************************************************************************
	 * Move byte pointer by offset bytes
	 * @param offset 
	 ***************************************************************************/
	public void seek(int offset){
		counter = offset;
	}
	
	
	/**
	 * Showing in hex file format
	 * Debug functions
	 */
	public String toFile(){
		String str = "\r";
		int checksum = 0;
		for(int j=0; j<buffer.length; j++){
			//Start of line, add header, reset checksum
			if(j%16 == 0){
				checksum = 0;
				str += "\n" + ":10";
				int lineAddr = (start_addr*2)+j;
				str += toWord(lineAddr); 
				str += "00";
				checksum += (0x10 + ((lineAddr >> 8) & 0xff) + (lineAddr & 0xff));
			}
			//add byte, update checksum
			str += toByte(buffer[j]);
			checksum += buffer[j];
			//End of line
			if(j%16 == 15){
				byte cksum = (byte)(0x0100 - (checksum & 0xff)); 
				str += toByte(cksum);				
			}			
//			if(j == 15)
//				str += "     0x" + Integer.toHexString(start_addr);
		}
		return str;
	}
	
	/**
	 * Showing a row of memory
	 * Debug functions
	 */
	public String toString(){
		String str = "Address: 0x" + Integer.toHexString(start_addr);
		for(int j=0; j<buffer.length; j++){
			if(j%16 == 0){
				str += "\n" + ":--------";			
			}
			str += toByte(buffer[j]);
		}
		str += "\n";
		return str;
	}
	
	/**
	 * Showing a 2 digit byte
	 * Debug functions
	 */	
	private String toByte(byte b){
		String str;
		if(b >= 0x00 && b<=0x0f)
			str = "0" + Integer.toHexString(b & 0xff);
		else
			str = Integer.toHexString(b & 0xff);
		str = str.toUpperCase();
		return str;
	}
	
	/**
	 * Showing a 4 digit byte
	 * Debug functions
	 */	
	private String toWord(int val){
		String str;
		//if more than 4 bytes, truncate
		if(val > 0xffff)
			val &= 0xffff;
		
		//Convert the four bytes
		if(val >= 0x0000 && val<=0x000f)
			str = "000" + Integer.toHexString(val);
		else if(val > 0x000f && val<=0x00ff)
			str = "00" + Integer.toHexString(val);
		else if(val > 0x00ff && val<=0x0fff)
			str = "0" + Integer.toHexString(val);
		else
			str = Integer.toHexString(val);
		str = str.toUpperCase();
		return str;
	}
}
