/**
 * @project		dsPicProgrammer Program
 * @description	dsPicProgrammer for Microchip dsPic with dsPicBootloader
 * @author			Dennis (dennis@amonics.com)
 ******************************************************************************   
 * Licensing:	GNU GENERAL PUBLIC LICENSE
 *				Version 2, June 1991
 *				Please refer to GNU license for details (license.txt)
 ******************************************************************************/

import java.io.IOException;
import java.text.ParseException;
import java.util.concurrent.TimeoutException;

import memory.*;
import device.*;
import exception.*;

/******************************************************************************
 * dsPicProgrammer Protocol
 * Handle receiving data and send data
 ******************************************************************************/
public class PicProtocol implements COMDataHandler, PicDefinition{
	
	COMPortManager comManager;			//Handle to COM port controller
	int command = INIT;					//Command status
	PicModel picModel = null;			//Store picModel
	PicFactory picFactory = null;		//Store pic memory data
	byte[] rx_buffer;					//Recieve buffer

//	Debug variables
	boolean ENABLE_SEND_DATA = true;	//false for debug
//	PicModel picModel = new Pic33FJ128GP306();
//	PicModel picModel = new Pic30F5011();
	
	/*********************************************************************
	 * Constructor
	 * @param comManager COM port controller
	 ********************************************************************/
	public PicProtocol(COMPortManager comManager) {
		this.comManager = comManager;				// set COM Port controller
	}
	
	
	/*************************************************************************
	 * Attempt to establish link to target board
	 * @function sends '0x55' every 10ms for 10sec, break if ack received
	 *************************************************************************/
	public void connect() throws TimeoutException, AckNakException, HardwareException{
		byte[] txData = new byte[1]; 
		txData[0] = ACK;			//Initial Code for Autobaud rate dectection
		command = INIT;
		try {
			this.waitRecAndTimeout(INIT, txData);	//ACK may be transmit periodically
		} catch (TimeoutException te) {
			throw new TimeoutException("Connection: " + te.getMessage());
		} catch (AckNakException ane) {
			throw new AckNakException(ane.getMessage() + " during connection." + getRxData());
		}
		System.out.println("Connection established.");
	}
	
	
	/************************************************************************
	 * Get bootloader version number from device
	 ************************************************************************/	
	public void getVersion() throws TimeoutException, AckNakException, HardwareException {
		byte[] txData = new byte[1]; 
		txData[0] = VERSION;		//Version Control command		
		command = VERSION;			//Command
		comManager.write(txData);
		try {
			this.waitRecAndTimeout(VERSION, null);
		} catch (TimeoutException te) {
			throw new TimeoutException("Get Version: " + te.getMessage());
		} catch (AckNakException ane) {
			throw new AckNakException(ane.getMessage() + " during get version." + getRxData());
		}
	}

	
	/************************************************************************
	 * Read device id from device
	 * +-- PROTOCOL:	READ + DEVID_ADDR_H + DEVID_ADDR_M + DEVID_ADDR_L 
	 * +-- DEVID at 0xFF0000 
	 * +-- DEVREV at 0xFF0002 (not implemented)
	 ************************************************************************/	
	public void readDEVID() throws TimeoutException, AckNakException, HardwareException {
		byte[] txData = new byte[4]; 
		txData[0] = READ;			//Read command
		txData[1] = (byte)0xff;		//HIGH address
		txData[2] = 0x00;			//MED address
		txData[3] = 0x00;			//LOW address
		command = READ;				//Command
		comManager.write(txData);
		try {
			this.waitRecAndTimeout(READ, null);
		} catch (TimeoutException te) {
			throw new TimeoutException("Read Device ID: " + te.getMessage());
		} catch (AckNakException ane) {
			throw new AckNakException(ane.getMessage() + " during read command." + getRxData());
		}
	}
	
	/************************************************************************
	 * Reset device and goto user program
	 ************************************************************************/	
	public void reset(){
		byte[] txData = new byte[1];
		txData[0] = RESET;
		command = RESET;
		comManager.write(txData);		
	}
	
	
	/**********************************************************************************
	 * Program memory using intel hex file
	 * +-- read record type line-by-line and divert to appropriate subfunctions
	 * @param rdFileIntelHex
	 * @throws AddrResException
	 * @throws HardwareException
	 * @throws IOException
	 * @throws ParseException
	 * @throws TimeoutException
	 * @throws AckNakException
	 **********************************************************************************/
	public void program(RdFileIntelHex rdFileIntelHex) 
		throws AddrResException, HardwareException, IOException, ParseException, TimeoutException, AckNakException {

		/*
		 * Effective address (intelHexAddr) = extLinearAddr + addr
		 * Effective address (dspicAddr) = intelHexAddr/2
		 */
		int extLinearAddr = 0;		//Store the most updated upper 16-bit extLinearAddr
		RdFileIntelHex.OneLineIntelHex oneLineIntelHex = rdFileIntelHex.oneLineIntelHex;
		while(picModel == null);	//wait for checkDeviceID() to finished
		this.picFactory = new PicFactory(picModel);	//Factory for memory data
		System.out.println(LINE_DIVIDER);
		System.out.println("Reading HEX file...");

		/*
		 * for each hex line, determine the type:
		 * if data, copy data to FlashRow
		 * if eof, program
		 * if extend linear address, update address
		 * otherwise, exception
		 */
		while (true) {
			oneLineIntelHex.parseLine();				// parse one line intel hex
			switch(oneLineIntelHex.recordType){
			case TT_DATA:		//data
				picFactory.readDataLine(oneLineIntelHex, extLinearAddr);
				break;
			case TT_EOF:		//end-of-file
				System.out.println("Reading HEX file completed.");
				System.out.println(LINE_DIVIDER);
				this.showMemory();
				System.out.print("Please wait. Programming in progress ");
				this.progFlash();
				this.progEEPROM();
				//TODO: Problem in writing configuration register, so skip 
//				this.progConfig();					
				System.out.println("All Programming Done.");
				System.out.println(LINE_DIVIDER);
				return;
			case TT_ESA:		//extend segment address, not implemented
				throw new AddrResException("Unexpected Record Type: Extended segment address (0x02)");
			case TT_ELA:		//extend linear address
				extLinearAddr = (((int)oneLineIntelHex.data[0] & 0xff) << 24);  //e.g. 0x00 -> 0x00000000
				extLinearAddr +=(((int)oneLineIntelHex.data[1] & 0xff) << 16);  //e.g. 0xff -> 0x00ff0000
				break;
			default:
				throw new AddrResException("Unexpected Record Type: Unknown (0x" 
						+ Integer.toHexString((int)oneLineIntelHex.recordType) + ")");			
			}
		}
	}

	
	/**************************************************************************************
	 * Program Flash data to device
	 * @param picAddr
	 * @throws TimeoutException
	 * @throws AckNakException
	 * @throws HardwareException
	 ***************************************************************************************/
	private void progFlash()
		throws TimeoutException, AckNakException, HardwareException {
		
		/*
		 * sort and add padding rows
		 */
		picFactory.sortFlashList();
		picFactory.insertPadRow();
		
		/*
		 * Program each row by row
		 */
		PicBuffer aBuffer = null;
		for(int row=0; row<picFactory.flashList.size(); row++){
			MemFlash aRow = picFactory.flashList.get(row);
			aBuffer = new PicBuffer(this.picModel, aRow.getStartAddr(),
							picModel.series.getFlashRowSize(),
							WRITE);
			/*
			 * Copy data to buffer
			 */
			for(int data=0; data<aRow.buffer.length; data++){
				boolean full = aBuffer.addData(aRow.buffer[data]);
				/*
				 * When buffer is full, send
				 */
				if(full){
					sendData(aBuffer);
					aBuffer = null;
				}
			}
/*			
			if(aRow.getStartAddr() >= 0x0000 && aRow.getStartAddr() < 0x4000)
//		    if(aRow.getStartAddr() >= 0x4000 && aRow.getStartAddr() < 0x8000)
//			if(aRow.getStartAddr() >= 0x8000 && aRow.getStartAddr() < 0x12000)
				System.out.print(aRow.toFile());
*/			
			
			/*
			 * Refresh screen periodically
			 * "Programming in progress..."
			 */
			switch(row%30){
				case 0:	case 5:	case 10:
					System.out.print("\b");
					System.out.print(". ");
					break;
				case 15: case 20: case 25:
					System.out.print("\b\b");
					System.out.print(" ");					
					break;
			}
		}
		System.out.println();		
		/*
		 * Last row of memory
		 */
		if(aBuffer != null){
			while(!aBuffer.addData((byte)0x00)); //Fill buffer with zeros
			sendData(aBuffer);
			aBuffer = null;
		}
		/*
		 * Interact with user if Flash has been written
		 */
		if(picFactory.flashList.size() > 0)
			System.out.println("   Flash memory done.");
	}
	
	/**************************************************************************************
	 * Program EEPROM data to device
	 * @param picAddr
	 * @throws TimeoutException
	 * @throws AckNakException
	 * @throws HardwareException
	 ***************************************************************************************/
	private void progEEPROM() 
		throws TimeoutException, AckNakException, HardwareException {
		/*
		 * program only if has eeprom
		 */
		if(this.picModel.series.getClass() == Pic30F.class){
			/*
			 * Program each row by row
			 */
			PicBuffer aBuffer = null;
			for(int row=0; row<picFactory.eepromList.size(); row++){
				MemEEPROM aRow = picFactory.eepromList.get(row);
				aBuffer = new PicBuffer(this.picModel, aRow.getStartAddr(),
							((Pic30F)picModel.series).getEEPROMRowSize(),
							WRITE);	
				/*
				 * Copy data to buffer
				 */
				for(int data=0; data<aRow.buffer.length; data++){
					boolean full = aBuffer.addData(aRow.buffer[data]);
					/*
					 * When buffer is full, send
					 */
					if(full){
						sendData(aBuffer);
						aBuffer = null;
					}
				}
			}
			/*
			 * Last row of memory
			 */
			if(aBuffer != null){
				while(!aBuffer.addData((byte)0x00)); //Fill buffer with zeros
				sendData(aBuffer);
				aBuffer = null;
			}
			
			/*
			 * Interact with user, EEPROM has been written 
			 */
			if(picFactory.eepromList.size() > 0)
				System.out.println("   EEPROM done.");
		}
	}

	
	/**************************************************************************************
	 * Program config data to device
	 * @param picAddr
	 * @throws TimeoutException
	 * @throws AckNakException
	 * @throws HardwareException
	 ***************************************************************************************/
	public void progConfig() 
		throws TimeoutException, AckNakException, HardwareException {
		/*
		 * Program each row by row
		 */
		PicBuffer aBuffer = null;
		for(int row=0; row<picFactory.configList.size(); row++){
			MemConfig aRow = picFactory.configList.get(row);
			aBuffer = new PicBuffer(this.picModel, aRow.getStartAddr(),
							picModel.series.getConfigRowSize(),
							WRITE);	
			/*
			 * Copy data to buffer
			 */
			for(int data=0; data<aRow.buffer.length; data++){
				boolean full = aBuffer.addData(aRow.buffer[data]);
				/*
				 * When buffer is full, send
				 */
				if(full){
					sendData(aBuffer);
					aBuffer = null;
				}
			}
		}
		/*
		 * Interact with user, config register has been written 
		 */
		if(picFactory.configList.size() > 0)
			System.out.println("   Configuration Register done.");
	}

	
	/*************************************************************************************
	 * Sending row/page of data
	 * @param currentBuffer
	 * @throws TimeoutException
	 * @throws AckNakException
	 * @throws Exception
	 *************************************************************************************/	
	public void sendData(PicBuffer currentBuffer) 
		throws TimeoutException, AckNakException, HardwareException{
		if(ENABLE_SEND_DATA){
			command = WRITE;
			comManager.write(currentBuffer.buffer);
			try {
				waitRecAndTimeout(WRITE, null);
			} catch (TimeoutException te) {
				throw new TimeoutException("Program: " + te.getMessage());
			} catch (AckNakException ane) {
				throw new AckNakException(ane.getMessage() + " when programming chip." + getRxData() );
			}					
		}	
	}
	
	
	/********************************************************************************
	 * Wait for Receive data and Timeout
	 * @param constCmd
	 * @param pdata
	 * @throws TimeoutException
	 * @throws AckNakException
	 *******************************************************************************/
	private void waitRecAndTimeout(int constCmd, byte[] pdata) 
		throws TimeoutException, AckNakException, HardwareException{
		try {
			int timeoutCnt = 0;				// setup timeout counter
			while (command==constCmd) {		//Do when command has not been changed, i.e. nothing has been received
				if(pdata!=null)	comManager.write(pdata);	//Periodic send data if needed
				Thread.sleep( SLEEP_TIME );
				timeoutCnt++;
				if (timeoutCnt>=TIMEOUT/SLEEP_TIME) throw new TimeoutException("Timeout");
			}
		} catch (InterruptedException e) {System.out.println(e);}
		//command has changed, if received command not valid, stop programmer
		if (command!=ACK){
			if(command==UNKNOWN_DEVICE)
				throw new HardwareException("Unknown device found. Programming abort.");
			else
				throw new AckNakException("Received NACK");
		}
	}
	
	
	/********************************************************************************
	 * Receive raw data, i.e. return command from device
	 * +-- Expected ACK, otherwise consider NACK
	 * +-- For READ (reading device id), also perform checking id 
	 ********************************************************************************/	
	public void recRawData(){
		switch (command) {
		case INIT: 
			//program reach here because comManager.serialEvent(e) has received ACK
			command = ACK;		
			break;	
		case VERSION: 
			rx_buffer = this.receive(3);
			if(rx_buffer.length >2 && rx_buffer[2]==ACK){
				command = ACK;
		    	System.out.println("  Firmware Version: " + rx_buffer[0] + "." + rx_buffer[1]);				
			} else{
				command = NACK;
			}
			break;
		case WRITE:
			rx_buffer = this.receive(1);
			command = (rx_buffer[0]==ACK) ? ACK : NACK;
			break;	
		case READ:
			rx_buffer = this.receive(4);
			if(rx_buffer.length >3 && rx_buffer[3]==ACK ){
				command = ACK;
				System.out.println("  Device ID: 0x" +  Integer.toHexString((0xff & rx_buffer[0])) +  
							Integer.toHexString((0xff & rx_buffer[1])) + 
							Integer.toHexString((0xff & rx_buffer[2])) );
				this.checkDeviceID(rx_buffer);
			} else {
				command = NACK;
			}
		    break;	
		}
		for (int i = 0;i<20000;i++);		// some delay for avoid error
	}
	
	
	/****************************************************************************
	 * Receive a minimum of count bytes from COM port
	 * @param count number of bytes to receive
	 ****************************************************************************/
	private byte[] receive(int count) {
		int data_cnt = 0;
		byte[] ans = null;		//cumulative storage
		byte[] buffer = null;	//new storage

		while(data_cnt < count){
			buffer = this.rx();
			data_cnt += buffer.length;
			
			byte[] tmp = new byte[data_cnt];	//tmp storage
			int i=0, j=0;
			if(ans != null){
				for(i=0; i<ans.length; i++)
					tmp[i] = ans[i];				//copy old data
			}
			for(j=0; j<buffer.length; j++, i++)
				tmp[i] = buffer[j];				//copy new data
				
			ans = tmp;							//update cumulative data
		}
		return ans;
	}
	
	/****************************************************************************
	 * Receive data from COM port
	 ****************************************************************************/	
	private byte[] rx(){
		int num;
		byte[] buffer = null;
	    try {
			while (true){
				num = comManager.inputStream.available();
				if(num > 0){
					buffer = new byte[num];
					comManager.inputStream.read(buffer);
					break;
				}
			}
	    } catch (IOException e) {System.err.println(e);}
	    return buffer;		
	}
			
	/****************************************************************************
	 * Check if device is supported
	 * +-- add your supported device list in this function
	 * +-- DEVID can be found in dsPIC30/dsPIC33 Flash Programming Specification
	 ****************************************************************************/
	private void checkDeviceID(byte[] id){
		int devid = ( ( (int)id[0] << 16 ) & 0xff0000) +
					( ( (int)id[1] << 8  ) & 0x00ff00) + 
					(   (int)id[2]         & 0x0000ff);
    	if(devid==DSPIC30F5011){
    		//Device ID for dsPic30F5011 = 0x000080
    		this.picModel = new Pic30F5011();
    		System.out.println("  Found device: " + picModel.getName());
		} 
    	else if(devid==DSPIC33FJ128GP306){
       		//Device ID for dsPic33FJ128GP306 = 0x0000E5
    		this.picModel = new Pic33FJ128GP306();
    		System.out.println("  Found device: " + picModel.getName());		   		    		
    	}
    	//TODO: add your new device here
    	else{
    		command = UNKNOWN_DEVICE;		//Unknown Device, for aborting operation
    	}		
	}

	/****************************************************************************
	 * Convert Received Data to Hex String
	 ****************************************************************************/
	private String getRxData(){
		String msg = " ";
		//Display only if -i option has been set
		if(PicProgrammer.getFlag().contentEquals(PROG_FLAG)){
			msg = " :: ";
			for(int i=0; i<rx_buffer.length; i++){
				msg += (Integer.toHexString( (0xff & rx_buffer[i]) ) + " ");
			}
		}
		return msg;			
	}
	
	/*
	 * Print memory content to console
	 * Debug functions, to be deleted
	 */
	private void showMemory(){
		/*
		for(int i=0; i<picFactory.flashList.size(); i++){
			MemFlash flash = picFactory.flashList.get(i);
			System.out.println(flash.toString());
		}
		//*/
		/*
		for(int i=0; i<picFactory.eepromList.size(); i++){
			MemEEPROM eeprom = picFactory.eepromList.get(i);
			System.out.println(eeprom.toString());
		}
		//*/
		/*
		for(int i=0; i<picFactory.configList.size(); i++){
			MemConfig config = picFactory.configList.get(i);
			System.out.println(config.toString());
		}
		//*/
	}
}
