/**
 * @project		dsPicProgrammer Program
 * @description	dsPicProgrammer for Microchip dsPic with dsPicBootloader
 * @author			Dennis (dennis@amonics.com)
 ******************************************************************************   
 * Licensing:	GNU GENERAL PUBLIC LICENSE
 *				Version 2, June 1991
 *				Please refer to GNU license for details (license.txt)
 ******************************************************************************/

import java.io.IOException;
import java.text.ParseException;
import java.util.concurrent.TimeoutException;

import gnu.io.NoSuchPortException;
import gnu.io.UnsupportedCommOperationException;

import exception.AckNakException;
import exception.AddrResException;
import exception.HardwareException;

/******************************************************************************
 * Rationale of dsPicProgrammer:
 * 
 * +-------------------+             +-------------------+
 * |  dsPicProgrammer  |             |  dsPicBootloader  |
 * +-------------------+             +-------------------+
 * |       PC          |             |   Target dsPic    |
 * +-------------------+             +-------------------+
 * |     COM PORT      |=============|       UART        |
 * +-------------------+             +-------------------+
 * 
 * Operation
 * =========
 * 1) dsPicProgrammer sends a Break character (pull TX to low logic, normally high)
 * 2) User program on dsPic detects the break character and reset the chip
 * 3) dsPicProgrammer begins communication with dsPicBootloader (sending 0x55)
 * 4) dsPicProgrammer gets version information from dsPicBootloader
 * 5) dsPicProgrammer reads the hex file
 * 6) dsPicProgrammer sends data to dsPicBootloader
 ******************************************************************************/

/******************************************************************************
 * Entry Class for dsPicProgrammer
 ******************************************************************************
 * Functions:
 * +-- Using serial port (ttyS0 | COM1) to communicate with dsPicBootloader
 * +-- Program an Intel Hex File to target device
 * +-- User can select baudrate
 ******************************************************************************
 * External classes:
 * +-- COMPortManager: Interface data stream with COM port
 * +-- RdFileIntelHex: Interface with data stream from HEX file
 * +-- PicProtocol:	   Interface with dsPicBootloader
 ******************************************************************************/
public class PicProgrammer implements PicDefinition {

	static COMPortManager comManager = null;
	static String arg_flag = "null";
	static String arg_serial_port = "null";
	static String arg_baudrate = "null";
	static String arg_file = "null";
	static RdFileIntelHex rdFileIntelHex = null;
	
	/************************************************************************
	 * main()
	 ************************************************************************/
	public static void main(String[] args) {
		
		handleCommandLineInput(args);
		
		checkHexExist();
		
		openSerialPort();
		
		/*
		 * Start Communication
		 */
		PicProtocol picProtocol = new PicProtocol(comManager);	// protocol
		comManager.setCOMDataHandler(picProtocol);
		try {
			setBaudRate();

			System.out.println(LINE_DIVIDER);
			System.out.println(PROG_NAME + " Version = " + PROG_VERSION);		
			
			if(!arg_flag.contentEquals(PROG_FLAG)){
				System.out.println(LINE_DIVIDER);
				System.out.println("Your target board is about to reset.");
				System.out.println("The existing program will be overwritten.");
				System.out.print("Are you sure to proceed? (Y/N) ");
				int tmp = System.in.read ();
				char c = (char) tmp;
				if(c == 'y' || c == 'Y'){
					startCommunication(picProtocol);
				}
				else{
					System.out.println("Operation abort.");
				}				
			}
			else{
				startCommunication(picProtocol);
			}
			//--------------------------------------------------------------
		} catch (IOException ioe) {
			System.err.println(ioe);
		} catch (ParseException pe) {
			System.err.println(pe);
	    } catch (UnsupportedCommOperationException ucoe) {
			System.err.println(ucoe);
		} catch (TimeoutException te) {
			System.err.println(te);
		} catch (AckNakException ane) {
			System.err.println(ane);
		} catch (AddrResException are){
			System.err.println(are);
		} catch (HardwareException he){
			System.err.println(he);			
		} catch (Exception e){
			System.err.println("Unknown error: " + e.getMessage());
			e.printStackTrace();
		} finally {								
			/*
			 * After work, close resources 
			 */
			if (rdFileIntelHex.oneLineIntelHex != null) {
				try {
					rdFileIntelHex.oneLineIntelHex.close();	//close file
				} catch (IOException ioe) {
					System.err.println("Error in closing Intel Hex file.");
					System.err.println(ioe);
				}
			}
			if (comManager != null) {
				comManager.close();		//close com port
				System.out.println("COM port closed successfully.");
			}
		} 
		
		/*
		 * Exiting
		 */
		System.out.println("Thank you for using " + PROG_NAME  + "."); 
		System.out.println("Bye Bye!");
		System.exit(0);
		
	} //end main
	
	/*
	 * Read Command Line Arguments 
	 */
	private static void handleCommandLineInput(String[] args){
		if(args.length == 3){
			arg_serial_port = args[0];
			arg_baudrate = args[1];
			arg_file = args[2];
		}
		else if(args.length == 4){
			arg_flag = args[0];
			arg_serial_port = args[1];
			arg_baudrate = args[2];
			arg_file = args[3];			
			if(!arg_flag.contentEquals(PROG_FLAG)){
				System.err.println("Usage of " + PROG_NAME + ":");
				System.err.println("	java -jar " + PROG_NAME + ".jar" +
											" /dev/ttyS0|COM1 19200 foo.hex");
				System.exit(0);				
			}
		}
		//Prompt for help when inputs are invalid
		else {
			System.err.println("Usage of " + PROG_NAME + ":");
			System.err.println("	java -jar " + PROG_NAME + ".jar" +
										" /dev/ttyS0|COM1 19200 foo.hex");
			System.exit(0);
		}		
	}

	/*
	 * Test for existence of Intel Hex File 
	 */
	private static void checkHexExist(){
		try {									
			rdFileIntelHex = new RdFileIntelHex(arg_file);
		} catch (Exception e) {
			System.err.println(e);
			System.exit(1);
		}		
	}

	/*
	 * Open COM port
	 */
	private static void openSerialPort(){
		try {									
			comManager = new COMPortManager(arg_serial_port);
		} catch (NoSuchPortException nspe) {
			System.err.println(nspe + ": Choosing COM Port Error");
			System.exit(1);
		} catch (Exception e) {
			System.err.println(e);
			System.exit(1);
		}		
	}

	/*
	 * Set Baudrate
	 */
	private static void setBaudRate() throws UnsupportedCommOperationException{
		int baudrate = Integer.parseInt(arg_baudrate);
		if(baudrate < 9600 || baudrate > 57600) {
			comManager.configPort(DEFAULT_BAUDRATE);

			System.out.println(LINE_DIVIDER);
			System.out.println("WARNING: Unsupported baudrate. ");
			System.out.println("         Using default baudrate of " + DEFAULT_BAUDRATE + "bps");
		} else {
			comManager.configPort(baudrate);
		}		
	}
	
	/*
	 * Procedure to start communication 
	 */
	private static void startCommunication(PicProtocol picProtocol) 
		throws IOException, TimeoutException, AckNakException, HardwareException, 
					ParseException, AddrResException
	{
		System.out.println(LINE_DIVIDER);
		System.out.println("If your target board fails to respond before ");
		System.out.println("timeout occurs, please restart your target ");
		System.out.println("manually and try again.");
		System.out.println(LINE_DIVIDER);

		comManager.sendBreak(BREAK_TIME);		//Send a break character to reset the target

		picProtocol.connect();					//Connect with dsPicBootloader
		picProtocol.getVersion();				//Get Bootloader version number
		picProtocol.readDEVID();				//Read device ID
		picProtocol.program(rdFileIntelHex);	//Program the device
		picProtocol.reset();					//Reset device and goto user's program		
	}
	
	/*
	 * Get flag
	 */
	public static String getFlag()
	{
		return arg_flag;
	}
}
