/**
 * @project		dsPicProgrammer Program
 * @description	dsPicProgrammer for Microchip dsPic with dsPicBootloader
 * @author			Dennis (dennis@amonics.com)
 ******************************************************************************   
 * Licensing:	GNU GENERAL PUBLIC LICENSE
 *				Version 2, June 1991
 *				Please refer to GNU license for details (license.txt)
 ******************************************************************************/

import java.util.ArrayList;
import java.util.List;

import memory.MemConfig;
import memory.MemEEPROM;
import memory.MemFlash;

import device.Pic30F;
import device.PicModel;

import exception.HardwareException;
import exception.AddrResException;

/******************************************************************************
 * Factory for pic memory
 ******************************************************************************/
public class PicFactory {
	
	PicModel picModel = null;
	
	List<MemFlash> flashList = new ArrayList<MemFlash>(); 
	List<MemEEPROM> eepromList = new ArrayList<MemEEPROM>();
	List<MemConfig> configList = new ArrayList<MemConfig>();

	/*********************************************************************
	 * Constructor
	 * @param model PicModel information
	 ********************************************************************/	
	public PicFactory(PicModel model){
		this.picModel = model;
	}

	
	/********************************************************************************
	 * Parse address, determine memory allocation and divert to appropriate subfunction 
	 * @param oneLineIntelHex
	 * @param extLinearAddr
	 * @throws AddrResException unsupported pic address
	 * @throws HardwareException pic does not have eeprom
	 ********************************************************************************/
	public void readDataLine(RdFileIntelHex.OneLineIntelHex oneLineIntelHex, int extLinearAddr) 
		throws HardwareException, AddrResException {
		
		/*
		 * Parse picAddr
		 * e.g. (0x00ff0000 + 0xf800)/2 = (0x00fff800)/2 = 0x7ffc00 -> eeprom
		 */
		int picAddr = (extLinearAddr + oneLineIntelHex.addr)/2;
		
		/*
		 * Use High Byte to determine memory type
		 */
		switch((picAddr>>16)& 0xff){
			case 0x00: 
			case 0x01: 	
				copyFlash(oneLineIntelHex, picAddr); 
				break;
			case 0x7f:
				if(this.picModel.series.getClass() == Pic30F.class){
					copyEEPROM(oneLineIntelHex, picAddr); 
					break;
				} else{
					throw new AddrResException("Unexpected Pic Address: Unknown (0x" + Integer.toHexString(picAddr) + ")");
				}
			case 0xf8:	
				copyConfig(oneLineIntelHex, picAddr); 
				break;
			default:
				throw new AddrResException("Unexpected Pic Address: Unknown (0x" + Integer.toHexString(picAddr) + ")");	
		}
	}

	
	/**************************************************************************************
	 * Copy one line of flash data to RAM
	 * @param oneLineIntelHex
	 * @param picAddr
	 * @throws AddrResException attempt to overwrite bootloader
	 ***************************************************************************************/
	private void copyFlash(RdFileIntelHex.OneLineIntelHex oneLineIntelHex, int picAddr) throws AddrResException{
		
		/*
		 * Verify reset vector address points to default address
		 * Create memory space for 2 ivt rows, 2 avt rows, and 1st upc row
		 */
		if(picAddr == 0){	
			verifyResetAddr(oneLineIntelHex);
			//Create Default rows
			for(int i=0, addr=0; i<5; i++){
				MemFlash aRow = new MemFlash(this.picModel, addr); 				
				flashList.add(aRow);
				addr += this.picModel.series.getFlashRowSize()/2;		//each address is 16-bit
			}
		}
			
		/*
		 * Protect bootloader section from writing
		 */
		else if(  	picAddr >= this.picModel.getBLStartAddr()
					&& picAddr < this.picModel.getBLEndAddr()		){
			throw new AddrResException("Attempting to overwrite bootloader... Execution abort.");
		}
		
		/*
		 * Copying IVT data (including reset vector)
		 */
		else if(	picAddr >= this.picModel.series.getIVTAddr()
					&& picAddr <  this.picModel.series.getAVTAddr()){
			copyIVT(oneLineIntelHex, picAddr);	
		} 

		/*
		 * Copying AVT data (including alternate reset vector)
		 */
		else if(	picAddr >= this.picModel.series.getAVTAddr()
					&& picAddr < this.picModel.series.getUPCAddr()){
			copyAVT(oneLineIntelHex, picAddr);
		}
		
		/*
		 * Copying User Program Code data (including reset vector)
		 */
		else{
			copyUPC(oneLineIntelHex, picAddr);
		}
	}

	
	/**************************************************************************************
	 * Check whether reset address at default position 0x100 (dspic30) or 0x200 (dspic33)
	 * @param oneLineIntelHex
	 * @throws AddrResException reset vector does not point to user code start address
	 ***************************************************************************************/
	private void verifyResetAddr(RdFileIntelHex.OneLineIntelHex oneLineIntelHex) 
		throws AddrResException{
		
		int resetAddr = 0;
		resetAddr = (int) (0x00ff & (byte)oneLineIntelHex.data[0]);
		resetAddr = resetAddr + (((int)oneLineIntelHex.data[1] & 0x00ff) << 8);
		if(resetAddr != this.picModel.series.getUPCAddr())
			throw new AddrResException("Reset vector address not at 0x" + 
								Integer.toHexString(this.picModel.series.getUPCAddr()) + 
								". Execution abort.");
	}

	
	/**************************************************************************************
	 * Copy IVT Data to Rows 1 and 2
	 * @param oneLineIntelHex
	 * @param picAddr
	 ***************************************************************************************/
	private void copyIVT(RdFileIntelHex.OneLineIntelHex oneLineIntelHex, int picAddr){
		
		MemFlash ivt = (picAddr < this.picModel.series.getFlashRowSize()/2)? 
							flashList.get(0) : 
							flashList.get(1); 
		/*
		 * Set Reset Vector to Bootloader Address
		 * 		0x000000	GOTO instruction
		 * 		0x000002	Reset address
		 * e.g.	0xhhmmll: Fill first 4 words as llmm 0400 hh00 0000
		 *  	0x00AE00: Fill first 4 words as 00ae 0400 0000 0000
		 * 		0X015000: Fill first 4 words as 0050 0400 0100 0000
		 */
		if(picAddr == this.picModel.series.getIVTAddr()){
			int bl_addr = this.picModel.getBLStartAddr();
			
			ivt.addData( (byte)(bl_addr) );			//ll 
			ivt.addData( (byte)(bl_addr >> 8) );	//mm
			ivt.addData( (byte) 0x04); 				//04	goto
			ivt.addData( (byte) 0x00);				//phathom byte
			ivt.addData( (byte)(bl_addr >> 16) );	//hh
			ivt.addData( (byte) 0x00);				//reserve
			ivt.addData( (byte) 0x00);				//reserve
			ivt.addData( (byte) 0x00);				//phathom byte
		}
		//-----------------------------------------------------------------
		for(int i=0; i<oneLineIntelHex.dataLen; i++){
			if(!ivt.addData(oneLineIntelHex.data[i])){
				ivt = flashList.get(1);
				ivt.addData(oneLineIntelHex.data[i]);		
			}
		}
	}

	
	/**************************************************************************************
	 * Copy AVT Data to Rows 3 and 4
	 * @param oneLineIntelHex
	 * @param picAddr
	 ***************************************************************************************/
	private void copyAVT(RdFileIntelHex.OneLineIntelHex oneLineIntelHex, int picAddr){
		
		MemFlash avt = (picAddr < this.picModel.series.getFlashRowSize()/2 + this.picModel.series.getFlashRowSize())? 
						flashList.get(2) : 
						flashList.get(3);
		/*
		 * fill first 4 words as 0000 0000 0000 0000
		 */
		if(picAddr == this.picModel.series.getAVTAddr()){
			for(int i=0; i<8;i++) avt.addData((byte)0x00);
		}
		//-----------------------------------------------------------------
		for(int i=0; i<oneLineIntelHex.dataLen; i++){
			if(!avt.addData(oneLineIntelHex.data[i])){
				avt = flashList.get(3);
				avt.addData(oneLineIntelHex.data[i]);					
			}
		}				
	}

	/**************************************************************************************
	 * Copy User Program Code to Row i, created as needed
	 * @param oneLineIntelHex
	 * @param picAddr
	 * @throws AddrResException address does not match a start row address
	 ***************************************************************************************/
	private void copyUPC(RdFileIntelHex.OneLineIntelHex oneLineIntelHex, int picAddr) throws AddrResException{
		/*
		 * Look for a suitable row in FlashList
		 */
		MemFlash upc = null;
		boolean found = false;
		for(int j=0; j<flashList.size(); j++){
			upc = flashList.get(j);
			if( (picAddr >= upc.getStartAddr()) && 
					(picAddr < upc.getStartAddr() + picModel.series.getFlashRowSize()/2) ){
				found = true;
				break;
			}
		}
		/*
		 * Found a suitable row
		 */
		if(found){
			/*
			 * Address is continuous, 
			 * 	Copy byte-by-byte. When full, create a new row and continue 
			 */
			if(picAddr == upc.getNextPosition())
			{
				for(int i=0; i<oneLineIntelHex.dataLen; i++){
					if(!upc.addData(oneLineIntelHex.data[i])){
						//Buffer is full, before creating a new row, search again for a suitable row  
						boolean fnd = false;
						for(int j=0; j<flashList.size(); j++){
							upc = flashList.get(j);
							if( (picAddr+i/2 >= upc.getStartAddr()) && 
									(picAddr+i/2 < upc.getStartAddr() + picModel.series.getFlashRowSize()/2) ){
								fnd = true;
								break;
							}
						}
						if(!fnd)
						{
							upc = new MemFlash(this.picModel, picAddr+i/2);							
							flashList.add(upc);
						}
						upc.seek(0);
						upc.addData(oneLineIntelHex.data[i]);
					}							
				}
			}
			/*
			 * Address is discontinous,
			 *  Move pointer to desire position
			 */
			else
			{
				upc.seek(2*(picAddr - upc.getStartAddr()));
				for(int i=0; i<oneLineIntelHex.dataLen; i++){
					if(!upc.addData(oneLineIntelHex.data[i])){
						throw new AddrResException("Flash Row Buffer pointer overflow.");
					}							
				}
			}
		}
		/*
		 * No suitable row
		 */
		else{
			upc = new MemFlash(this.picModel, picAddr);
			flashList.add(upc);
			for(int i=0; i<oneLineIntelHex.dataLen; i++){
				if(!upc.addData(oneLineIntelHex.data[i])){
					//Buffer is full, before creating a new row, search again for a suitable row  
					boolean fnd = false;
					for(int j=0; j<flashList.size(); j++){
						upc = flashList.get(j);
						if( (picAddr+i/2 >= upc.getStartAddr()) && 
								(picAddr+i/2 < upc.getStartAddr() + picModel.series.getFlashRowSize()/2) ){
							fnd = true;
							break;
						}
					}
					if(!fnd)
					{
						upc = new MemFlash(this.picModel, picAddr+i/2);							
						flashList.add(upc);
					}
					upc.seek(0);
					upc.addData(oneLineIntelHex.data[i]);
				}							
			}
		}
	}
	
	/********************************************************************************
	 * Handle one line of EEPROM data
	 * @param oneLineIntelHex
	 * @param picAddr
	 * @throws HardwareException picModel does not support eeprom
	 * @throws AddrResException address does not match a start row address
	 ********************************************************************************/
	private void copyEEPROM(RdFileIntelHex.OneLineIntelHex oneLineIntelHex, int picAddr) 
		throws HardwareException, AddrResException{
		
		MemEEPROM eeprom = null;
		/*
		 * eepromList is empty, create new row
		 */
		if(eepromList.size() == 0){
			eeprom = new MemEEPROM(this.picModel, picAddr);
			eepromList.add(eeprom);
		} 
		/*
		 * eepromList is not empty, get last row
		 */
		else{
			eeprom = eepromList.get(eepromList.size()-1);
		}
		/*
		 *  Copy byte-by-byte
		 * When full, create a new row and continue
		 */
		for(int i=0; i<oneLineIntelHex.dataLen; i++){		
			if(!eeprom.addData(oneLineIntelHex.data[i])){
				eeprom = new MemEEPROM(this.picModel, picAddr+i/2);
				eeprom.addData(oneLineIntelHex.data[i]);
				eepromList.add(eeprom);
			}
		}		
	}
	

	/*************************************************************************************
	 * Handle one line of Configuration Register data
	 * @param oneLineIntelHex
	 * @param picAddr
	 *************************************************************************************/
	private void copyConfig(RdFileIntelHex.OneLineIntelHex oneLineIntelHex, int picAddr) {
		
		MemConfig config = null;
		/*
		 * eepromList is empty, create new row
		 */
		if(configList.size() == 0){
			config = new MemConfig(this.picModel, picAddr);
			configList.add(config);
		} 
		/*
		 * eepromList is not empty, get last row
		 */
		else{
			config = configList.get(configList.size()-1);
		}
		/*
		 *  Copy byte-by-byte
		 * When full, create a new row and continue
		 */
		for(int i=0; i<oneLineIntelHex.dataLen; i++){		
			if(!config.addData(oneLineIntelHex.data[i])){
				config = new MemConfig(this.picModel, picAddr+i/2);
				config.addData(oneLineIntelHex.data[i]);
				configList.add(config);
			}
		}		
	}
	
	
	/**
	 * Sort the list in ascending order of start address 
	 * 
	 */
	public void sortFlashList()
	{
		/*
		 * Bubble Sort
		 */
		boolean done = true;
		do{
			done = true;
			for(int i=0; i<this.flashList.size()-1; i++){
				MemFlash RowA = this.flashList.get(i);
				MemFlash RowB = this.flashList.get(i+1);
				//Swap if A > B
				if(RowA.getStartAddr() > RowB.getStartAddr()){
					this.flashList.remove(i);
					this.flashList.add(i+1, RowA);
					done = false;					
				}
			}			
		} while(!done);		
	}
	
	
	/**
	 * Insert padding rows if breaks are founded
	 * Must execute sortFlashList first
	 */
	public void insertPadRow()
	{
		for(int i=0; i<this.flashList.size()-1; i++){
			MemFlash RowA = this.flashList.get(i);
			MemFlash RowB = this.flashList.get(i+1);
			int nextAddr = (RowA.getStartAddr() + this.picModel.series.getFlashRowSize()/2) ;
			if(nextAddr != RowB.getStartAddr()){
				MemFlash row = new MemFlash(this.picModel, nextAddr);
				this.flashList.add(i+1, row);
			}
		}
	}

}
