/**
 * @project		dsPicProgrammer Program
 * @description	dsPicProgrammer for Microchip dsPic with dsPicBootloader
 * @author			Dennis (dennis@amonics.com)
 ******************************************************************************   
 * Licensing:	GNU GENERAL PUBLIC LICENSE
 *				Version 2, June 1991
 *				Please refer to GNU license for details (license.txt)
 ******************************************************************************/

import device.*;

/*****************************************************************************
 * Buffer for page/row erase and write operation
 *****************************************************************************/
public class PicBuffer implements PicDefinition {
	
	public int dataSize = 0;
	public byte[] buffer = null;
	public byte checksum = 0;
	public int index = 0;
	
	/*************************************************************************
	 * Constructor
	 * @param address pic address
	 * @param size data packet size
	 * @param command write command
	 *************************************************************************/
	public PicBuffer(PicModel model, int address, int size, byte command){
		//dsPic30F: use 1 byte of data len
		if(model.series.getClass() == Pic30F.class){
			dataSize = size;
			buffer = new byte[1 + 3 + 1 + size + 1];	
			buffer[index++] = command;			// First Byte: Command, not counted in checksum
			this.addData((byte)(address>>16));	// High byte
			this.addData((byte)(address>>8));	// Medium byte
			this.addData((byte)address);		// Low byte
			this.addData((byte)(size+1));		//#data + checksum byte
		}
		//dsPic33F: Use 2 bytes for data len
		else{
			dataSize = size;
			buffer = new byte[1 + 3 + 2 + size + 1];	
			buffer[index++] = command;			// First Byte: Command, not counted in checksum
			this.addData((byte)(address>>16));	// High byte
			this.addData((byte)(address>>8));	// Medium byte
			this.addData((byte)address);		// Low byte
			this.addData((byte)((size+1)>>8));	//#data + checksum byte (High byte)			
			this.addData((byte)(size+1));		//#data + checksum byte (Low Byte)	
		}
	}
	
	
	/***************************************************************************
	 * Add data to buffer and calculate checksum
	 * @param data add byte to buffer
	 * @return buffer is full
	 ***************************************************************************/
	public boolean addData(byte data){
		buffer[index] = data;
		checksum += data;						//checksum ignore command
		index++;
		if(index == buffer.length-1){
			checksum = (byte)(0x100 - checksum); //Take 2's complement
			buffer[index] = checksum;
			index++;
		}
		return (index < buffer.length-1)? false : true;
	}
	
	
	/****************************************************************************
	 * Convert buffer to string (for debugging)
	 ****************************************************************************/
	public String toString(){
		String str = " ";
		for(int i=0; i<buffer.length; i++){
			str += Integer.toHexString((0xff & buffer[i])) + " ";
		}
		return str;
	}	
}
