/**
 * @project		dsPicProgrammer Program
 * @description	dsPicProgrammer for Microchip dsPic with dsPicBootloader
 * @author			Dennis (dennis@amonics.com)
 ******************************************************************************   
 * Licensing:	GNU GENERAL PUBLIC LICENSE
 *				Version 2, June 1991
 *				Please refer to GNU license for details (license.txt)
 ******************************************************************************/

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.TooManyListenersException;

import gnu.io.CommPortIdentifier;
import gnu.io.SerialPort;
import gnu.io.SerialPortEvent;
import gnu.io.SerialPortEventListener;
import gnu.io.NoSuchPortException;
import gnu.io.UnsupportedCommOperationException;

/******************************************************************************
 * Control COM Port access and operation
 * Objectives : send/receive byte by byte and init port
 * External dependencies:	RXTX drivers
 ******************************************************************************/
public class COMPortManager implements Runnable, SerialPortEventListener, PicDefinition {
	
	static CommPortIdentifier portId;	//store com port id object
	static SerialPort sport;			//store serial port object when portId is valid and not in use

	OutputStream outputStream;
	InputStream inputStream;

	Thread	uartRdWrThread;			//thread for running 
    COMDataHandler comDataHandler;	//Handle receiving data

    boolean error = true;			//assume there is error (e.g. break interrupt)
    
	/**************************************************************************
	 * Constructor
	 * @param port OS-dependent port name
	 * @Function Check for port existence -> Obtain port access from system ->
	 * Open input and out streams -> Configure baud rate and setting 8N1 ->
	 * Adding event listener -> Start Thread
	 **************************************************************************/
	public COMPortManager(String port) throws NoSuchPortException, Exception {
		
		portId = CommPortIdentifier.getPortIdentifier(port);	// choose port device
		
		/*
		 * Get access to COM port
		 */
		try {													
	    	sport = (SerialPort) portId.open(PROG_NAME, 2000); 
	    	//PROG_NAME is a owner name of this program
	    	//2000 is the timeout in (ms)
	    	//Will throws PortInUseException if port is being accessed by other programs
		} catch (Exception e) {
			System.out.println(e);
			throw new Exception(port + " is using by others");
		}

		/*
		 * Open I/O stream
		 */
		try {
			inputStream = sport.getInputStream();
			outputStream = sport.getOutputStream();
	    } catch (IOException e) {
	    	sport.close();
	    	throw e;
	    }
	    
	    /*
	     * Configure COM Port
	     */
	    try {
	    	this.configPort(DEFAULT_BAUDRATE);			//Set baud rate, 8N1
	    } catch (UnsupportedCommOperationException e) {
	    	sport.close();
	    	throw new Exception(port + " configuration error");
	    }
	    
	    /*
	     * Add EventListener for receive data events
	     */
	    try {
			sport.addEventListener(this);		//Add event listener to serial port
		} catch (TooManyListenersException e) {
	    	sport.close();
	    	throw e;
		}
		
		/*
		 * Start the thread
		 */
		for (int i = 0;i<10000;i++);			// some delay for avoid error
		sport.notifyOnDataAvailable(true);		
		uartRdWrThread = new Thread(this);		//create thread and start listening to receiver
		uartRdWrThread.start();
	}

	
	/********************************************************************************
	 * Set baud rate and 8N1
	 * @param baudrate baud rate
	 ********************************************************************************/
	public void configPort(int baudrate) throws UnsupportedCommOperationException {
		for (int i = 0;i<20000;i++);			// some delay for avoid error
    	sport.setSerialPortParams(baudrate, 
					       SerialPort.DATABITS_8, 
					       SerialPort.STOPBITS_1, 
					       SerialPort.PARITY_NONE);
	}

	
	/********************************************************************************
	 * Send Break character
	 * @param duration in milliseconds
	 ********************************************************************************/
	public void sendBreak(int duration){
		sport.sendBreak(duration);
	}
	
	
	/********************************************************************************
	 * Run this thread to receive COM Port Data
	 * @see java.lang.Runnable#run()
	 ********************************************************************************/
	public void run() {
		try {
		    Thread.sleep(20000);
		} catch (InterruptedException e) {}
	}

	
	/********************************************************************************
	 * Set data handler (to handle receive data)
	 * @param comDataHandler COM Port Data Handler for receiving data
	 ********************************************************************************/
	public void setCOMDataHandler(COMDataHandler comDataHandler) {
		this.comDataHandler = comDataHandler;
	}

	
	/********************************************************************************
	 * Event handler for COM port
	 * @param event events on COM port
	 ********************************************************************************/
	public void serialEvent(SerialPortEvent event) {
		switch (event.getEventType()) {
			//Ignore these events
			case SerialPortEvent.BI:		//Break Interrupt
			case SerialPortEvent.OE:		//Overrun Error
			case SerialPortEvent.FE:		//Framing Error
			case SerialPortEvent.PE:		//Parity Error
			case SerialPortEvent.CD:		//Carrier Detect
			case SerialPortEvent.CTS:		//Clear To Send
			case SerialPortEvent.DSR:		//Data Set Ready
			case SerialPortEvent.RI:		//Ring Indicator
			case SerialPortEvent.OUTPUT_BUFFER_EMPTY:
				break;
		    //If data is available, receive raw data and handle by uartRdWrDataHandle 
			case SerialPortEvent.DATA_AVAILABLE:
				if(error){
					int data;
					try {
						
						data = this.inputStream.read();

//						System.out.print(" " + Integer.toHexString(data));

						if(data == ACK  /*|| data == (int)0xd5*/){
							error = false;
						}

					} catch (IOException e) {System.err.println(e);}					
				} 
				if(!error){
					if (comDataHandler!=null) comDataHandler.recRawData();					
				}
				break;
		}
	}
	
	
	/*******************************************************************************
	 * Write data to COM port
	 * @param data array of data to transmit
	 *******************************************************************************/
	public void write(byte[] data) {
	    try {
			outputStream.write(data);
		} catch (IOException e) {
			System.out.println(e.toString());
		}
	}
	
	
	/********************************************************************************
	 * Close COM port (called by main)
	 * @see javax.comm.CommPort#close()
	 ********************************************************************************/
	public void close() {
		sport.close();
	}

}
